<?php

namespace App\Http\Controllers\Api\Admin\V1_0;

use App\Http\Controllers\Controller;
use App\Http\Utils\Miscellaneous;
use App\Http\Utils\MyLog;
use App\Models\Admin\Client;
use App\Models\Admin\DeliveryAddress;
use App\Models\Admin\Reseller;
use App\Models\Admin\ResellerCompany;
use App\User;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Hash;
use Validator;
use Log;

class ResellerCompanyController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        $query = ResellerCompany::query();

        $inputs = Input::get();
        foreach ($inputs as $key => $value) {
            if ($key != 'page' && $key != 'pagination_per_page') {
                if ($key == 'name' || $key == 'town')
                    $query->where('reseller_companies.' . $key,'like', '%'.$value.'%');
                else
                    $query->where('reseller_companies.' . $key,$value);
            }
        }
        $query->orderBy('client_id');
        $query->with('client', 'favourites');

        if (isset($inputs['pagination_per_page']))
            return $query->paginate($inputs['pagination_per_page']);
        else if (!isset($inputs['pagination_per_page']) && isset($inputs['page']))
            return $query->paginate();
        else if (!isset($inputs['pagination_per_page']) && !isset($inputs['page']))
            return $query->get();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        $warning = null;
        $status = null;

        $validator = Validator::make($request->all(), [
            'company.client_code' => 'required|min:3',
            'company.name' => 'required|min:3|max:60',
            'company.town' => 'required|min:3|max:50',
            'company.zip_code' => 'digits:5'
        ]);

        if ($validator->fails()){
            $status = 400;
            $warning = 'validation_error';
            $messages = $validator->errors()->all();
            MyLog::Warning($status, $warning, null, $messages);
            return response()->json(['error' => $warning, 'messages' => $messages], $status);
        }

        $client = Client::where('ERP_code', $request->input('company.client_code'))->first();
        if ($client != null) {
            $status = 400;
            $warning = 'already_exist';
            $messages = ['Client code already exists'];
            MyLog::Warning($status, $warning, null, $messages);
            return response()->json(['error' => $warning, 'messages' => $messages], $status);
        }

        DB::transaction(function () use ($request) {
            // Client information storage
            $client = new Client();
            $client->ERP_code = $request->input('company.client_code');
            $client->save();
            // Company information storage
            $company = new ResellerCompany();
            if ($request->has('company')) {
                foreach($request->company as $key => $value) {
                    if ($key != 'client_code')
                        $company[$key] = $value;
                }
                $company->client_id = $client->id;
                $company->save();
            }

            // Discount information storage
            if ($request->has('discounts')){
                $discounts_id_name_map = array();
                $discounts = $request->discounts;
                foreach ($discounts as $discount) {
                    //$discount = collect($discount);
                    //$company->discounts()->attach($discount);
                    $discount_id = $discount['discount_id'];
                    $rate = $discount['rate'];
                    $discount_value_map['rate'] = $rate;
                    $discounts_id_name_map[$discount_id] = $discount_value_map;
                }
                $company->discounts()->sync($discounts_id_name_map);
            }

            // Reseller information storage
            if ($request->has('resellers')) {
                $resellers = $request->resellers;
                foreach ($resellers as $reseller_input) {
                    //
                    $warning = null;
                    $status = null;
                    $input_user = $reseller_input['user'];
                    if ($input_user == null) {
                        $status = 404;
                        $warning = 'invalid_user';
                        $message = Lang::get('messages.' . $warning);
                        MyLog::Warning($status, $warning, $message);
                        return response()->json(['error' => $warning, 'message' => $message], $status);
                    }

                    // Individual information storage
                    $user_stored = User::where('email', $reseller_input['user']['login'])->first();
                    if ($user_stored != null)
                        $warning = 'user_already_exist_email';
                    if ($warning != null) {
                        $status = 409;
                        $message = Lang::get('messages.' . $warning);
                        MyLog::Warning($status, $warning, $message);
                        return response()->json(['error' => $warning, 'message' => $message], $status);
                    }

                    // User information storage
                    $user = new User();
                    if ($input_user != null) {
                        $user->email = $input_user['login'];
                        $user->password = Hash::make($input_user['password']);
                        $user->name = '';
                        $user->is_reseller = 1;
                        $user->verified = 1;
                        $user->save();
                    }

                    $reseller = new Reseller();
                    foreach($reseller_input as $key => $value) {
                        if ($key != 'user')
                            $reseller[$key] = $value;
                    }

                    $user->reseller()->save($reseller);
                    $company->resellers()->save($reseller);
                }
            }

            // Delivery Address information storage
            if ($request->has('delivery_addresses')){
                $deliveryAddresses = $request->delivery_addresses;
                foreach ($deliveryAddresses as $deliveryAddressInput) {
                    $deliveryAddress = new DeliveryAddress();
                    foreach ($deliveryAddressInput as $key => $value) {
                        $deliveryAddress[$key] = $value;
                    }
                    $company->deliveryAddresses()->save($deliveryAddress);
                }
            }

            Miscellaneous::incrementVersion();
        });
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
        $company = ResellerCompany::findOrFail($id);
        $company->client;
        $company->discounts;
        $resellers = $company->resellers;
        foreach ($resellers as $reseller)
            $reseller->user;
        $favourites = $company->favourites;
        foreach ($favourites as $favourite)
            $favourite->product;
        $company->deliveryAddresses;

        return $company;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
        $warning = null;
        $status = null;

        $validator = Validator::make($request->all(), [
            'company.client_code' => 'min:3|max:10',
            'company.name' => 'min:3|max:60',
            'company.town' => 'min:3|max:50',
            'company.zip_code' => 'digits:5'
        ]);

        if ($validator->fails()){
            $status = 400;
            $warning = 'validation_error';
            $messages = $validator->errors()->all();
            MyLog::Warning($status, $warning, null, $messages);
            return response()->json(['error' => $warning, 'messages' => $messages], $status);
        }


        $company = ResellerCompany::findOrFail($id);

        if ($request->has('company.client_code')) {
            $client = Client::where('ERP_code', $request->input('company.client_code'))->first();
            if ($client != null && $client->id != $company->client_id) {
                $status = 400;
                $warning = 'already_exist';
                $messages = ['Client code already exists'];
                MyLog::Warning($status, $warning, null, $messages);
                return response()->json(['error' => $warning, 'messages' => $messages], $status);
            }
        }

        DB::transaction(function () use ($request, $company) {
            // Dirty code to clean-up discount rates before changing the client association
            if ($request->has('company.client_code') && $request->has('discounts')) {
                $company->discounts()->detach();
                $company->save();
            }

            // Company information storage
            if ($request->has('company')) {
                foreach($request->company as $key => $value) {
                    if ($key != 'client_code')
                        $company[$key] = $value;
                }
                $client = $company->client;
                if ($request->has('company.client_code') &&
                    $client->ERP_code != $request->input('company.client_code')) {
                    $client->ERP_code = $request->input('company.client_code');
                    $client->save();
                }
                $company->save();
            }

            // Discount information storage
            if ($request->has('discounts')){
                $discounts_id_name_map = array();
                $discounts = $request->discounts;
                foreach ($discounts as $discount) {
                    //$discount = collect($discount);
                    //$company->discounts()->attach($discount);
                    $discount_id = $discount['discount_id'];
                    $rate = $discount['rate'];
                    $discount_value_map['rate'] = $rate;
                    $discounts_id_name_map[$discount_id] = $discount_value_map;
                }
                $company->discounts()->sync($discounts_id_name_map);
            }

            // Favourite information storage
            if ($request->has('favourites')){
                $favourites_input = $request->favourites;
                foreach ($favourites_input as $favourite_input) {
                    $favourite = $company->favourites()->findOrFail($favourite_input['id']);
                    $favourite->deactivated = $favourite_input['deactivated'];
                    if ($favourite->deactivated)
                        $favourite->delete();
                }
            }

            // Delivery Address information storage
            if ($request->has('delivery_addresses')){
                $deliveryAddressesInput = $request->delivery_addresses;
                foreach ($deliveryAddressesInput as $deliveryAddressInput) {
                    $deliveryAddress = null;
                    if (isset($deliveryAddressInput['id']))
                        $deliveryAddress = $company->deliveryAddresses()->find($deliveryAddressInput['id']);
                    if ($deliveryAddress == null)
                        $deliveryAddress = new DeliveryAddress();
                    foreach ($deliveryAddressInput as $key => $value) {
                        $deliveryAddress[$key] = $value;
                    }
                    $company->deliveryAddresses()->save($deliveryAddress);
                }
            }

            Miscellaneous::incrementVersion();
        });
    }
}
